#!/bin/bash

# BigGrep Rust Integration Test Suite
# Tests all BigGrep tools working together in realistic scenarios

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Test configuration
TEST_DIR="./test_integration"
DATA_DIR="$TEST_DIR/data"
INDEX_DIR="$TEST_DIR/indexes"
RESULTS_DIR="$TEST_DIR/results"
BINARY_DIR="./target/release"

# Test counters
TESTS_RUN=0
TESTS_PASSED=0
TESTS_FAILED=0

# Utility functions
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[PASS]${NC} $1"
}

print_failure() {
    echo -e "${RED}[FAIL]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

# Test result tracking
test_start() {
    local test_name="$1"
    TESTS_RUN=$((TESTS_RUN + 1))
    echo ""
    print_status "Running test: $test_name"
    echo "============================================"
}

test_pass() {
    local message="$1"
    TESTS_PASSED=$((TESTS_PASSED + 1))
    print_success "$message"
}

test_fail() {
    local message="$1"
    TESTS_FAILED=$((TESTS_FAILED + 1))
    print_failure "$message"
}

# Setup test environment
setup_test_environment() {
    print_status "Setting up test environment..."
    
    # Clean and create directories
    rm -rf "$TEST_DIR"
    mkdir -p "$DATA_DIR" "$INDEX_DIR" "$RESULTS_DIR"
    
    # Check if binaries exist
    if [ ! -x "$BINARY_DIR/rs-bgindex" ]; then
        print_error "Rust binaries not found. Building..."
        cargo build --release
        if [ ! -x "$BINARY_DIR/rs-bgindex" ]; then
            print_error "Failed to build Rust binaries"
            exit 1
        fi
    fi
    
    print_success "Test environment ready"
}

# Create test data files
create_test_data() {
    print_status "Creating test data files..."
    
    # Text files with various content
    cat > "$DATA_DIR/config.txt" << 'EOF'
# Configuration file
database_host=localhost
database_port=5432
api_key=sk_test_1234567890abcdef
secret_key=secret_abcdef123456
debug_mode=true
log_level=info

# User settings
username=admin
password=supersecret123
session_timeout=3600
EOF

    cat > "$DATA_DIR/logs.txt" << 'EOF'
2024-01-15 10:30:15 INFO Starting application
2024-01-15 10:30:16 DEBUG Loading configuration
2024-01-15 10:30:17 INFO Connecting to database
2024-01-15 10:30:18 ERROR Failed to connect: timeout
2024-01-15 10:30:19 WARNING Retrying connection
2024-01-15 10:30:20 INFO Connection established
2024-01-15 10:30:21 DEBUG User login: admin
2024-01-15 10:30:22 ERROR Invalid API key provided
2024-01-15 10:30:23 INFO Session created for user admin
2024-01-15 10:30:24 DEBUG Processing request
EOF

    cat > "$DATA_DIR/users.json" << 'EOF'
{
  "users": [
    {
      "id": 1,
      "username": "admin",
      "email": "admin@example.com",
      "api_key": "key_admin_123",
      "password": "hashed_password_1",
      "role": "administrator"
    },
    {
      "id": 2,
      "username": "user1",
      "email": "user1@example.com",
      "api_key": "key_user_456",
      "password": "hashed_password_2",
      "role": "user"
    },
    {
      "id": 3,
      "username": "test",
      "email": "test@example.com",
      "api_key": "key_test_789",
      "password": "hashed_password_3",
      "role": "tester"
    }
  ],
  "metadata": {
    "version": "1.0",
    "created": "2024-01-15T10:30:00Z"
  }
}
EOF

    cat > "$DATA_DIR/error.log" << 'EOF'
[2024-01-15 10:30:15] ERROR: Database connection failed
[2024-01-15 10:30:16] ERROR: Invalid credentials for user 'test'
[2024-01-15 10:30:17] WARNING: API rate limit exceeded
[2024-01-15 10:30:18] ERROR: Failed to parse configuration
[2024-01-15 10:30:19] ERROR: Permission denied accessing /etc/config
[2024-01-15 10:30:20] CRITICAL: Security breach detected
EOF

    # Binary file with specific pattern (ZIP file signature)
    printf '\x50\x4b\x03\x04\x14\x00\x00\x00\x08\x00' > "$DATA_DIR/binary_data.bin"
    
    # Large file for performance testing
    for i in {1..1000}; do
        echo "Line $i: This is test data with various keywords like password, api_key, secret, config, error, warning, debug, test, data" >> "$DATA_DIR/large.txt"
    done
    
    # Create additional test files
    for i in {1..50}; do
        cat > "$DATA_DIR/test_$i.txt" << EOF
Test file number $i
Contains sensitive information
Password: pass$i
API Key: key$i
Secret: secret$i
Config: config$i
EOF
    done
    
    # Create file list
    find "$DATA_DIR" -type f -exec echo {} \; > "$DATA_DIR/file_list.txt"
    
    print_success "Created $(find "$DATA_DIR" -type f | wc -l) test files"
}

# Test: Basic Index Building
test_basic_indexing() {
    test_start "Basic Index Building"
    
    # Build index
    if $BINARY_DIR/rs-bgindex -p "$INDEX_DIR/test_index" < "$DATA_DIR/file_list.txt" > "$RESULTS_DIR/index.log" 2>&1; then
        if [ -f "$INDEX_DIR/test_index.bgi" ]; then
            test_pass "Index file created successfully"
        else
            test_fail "Index file not created"
        fi
    else
        test_fail "Index building failed"
    fi
}

# Test: Basic Search
test_basic_search() {
    test_start "Basic Search Functionality"
    
    # Test ASCII search
    local results="$RESULTS_DIR/search_ascii.txt"
    if $BINARY_DIR/rs-bgsearch -a "password" -d "$INDEX_DIR" > "$results" 2>&1; then
        if grep -q "password" "$results"; then
            test_pass "ASCII search found expected results"
        else
            test_fail "ASCII search did not find expected results"
        fi
    else
        test_fail "ASCII search failed"
    fi
    
    # Test case sensitivity
    local case_results="$RESULTS_DIR/search_case.txt"
    if $BINARY_DIR/rs-bgsearch -a "PASSWORD" -d "$INDEX_DIR" > "$case_results" 2>&1; then
        if grep -q "PASSWORD" "$case_results"; then
            test_pass "Case-sensitive search works correctly"
        else
            test_warning "Case-sensitive search may need adjustment"
        fi
    fi
}

# Test: Binary Pattern Search
test_binary_search() {
    test_start "Binary Pattern Search"
    
    # Search for ZIP file signature
    local results="$RESULTS_DIR/search_binary.txt"
    if $BINARY_DIR/rs-bgsearch -b "504b0304" -d "$INDEX_DIR" > "$results" 2>&1; then
        if grep -q "binary_data.bin" "$results"; then
            test_pass "Binary pattern search found ZIP signature"
        else
            test_fail "Binary pattern search did not find expected signature"
        fi
    else
        test_fail "Binary pattern search failed"
    fi
}

# Test: Unicode Search
test_unicode_search() {
    test_start "Unicode Search"
    
    # Create file with Unicode content
    echo "Café résumé naïve" > "$DATA_DIR/unicode.txt"
    find "$DATA_DIR" -name "unicode.txt" -exec echo {} \; > "$DATA_DIR/unicode_list.txt"
    
    # Build small index for Unicode test
    $BINARY_DIR/rs-bgindex -p "$INDEX_DIR/unicode_index" < "$DATA_DIR/unicode_list.txt" > /dev/null 2>&1
    
    # Search for Unicode pattern
    local results="$RESULTS_DIR/search_unicode.txt"
    if $BINARY_DIR/rs-bgsearch -u "café" -d "$INDEX_DIR" > "$results" 2>&1; then
        if grep -q "unicode.txt" "$results"; then
            test_pass "Unicode search works correctly"
        else
            test_warning "Unicode search may need encoding adjustments"
        fi
    else
        test_fail "Unicode search failed"
    fi
}

# Test: Metadata Filtering
test_metadata_filtering() {
    test_start "Metadata Filtering"
    
    # Search with size filter
    local results="$RESULTS_DIR/search_filtered.txt"
    if $BINARY_DIR/rs-bgsearch -a "error" -d "$INDEX_DIR" -f "size>=100" > "$results" 2>&1; then
        if [ -s "$results" ]; then
            test_pass "Metadata filtering executed successfully"
        else
            test_warning "No results from metadata filtering (may be normal)"
        fi
    else
        test_fail "Metadata filtering failed"
    fi
}

# Test: Search with Verification
test_search_with_verification() {
    test_start "Search with Verification"
    
    # Search with verification enabled
    local results="$RESULTS_DIR/search_verified.txt"
    if timeout 30 $BINARY_DIR/rs-bgsearch -a "password" -d "$INDEX_DIR" -v > "$results" 2>&1; then
        if grep -q "verified\|password" "$results"; then
            test_pass "Search with verification works"
        else
            test_warning "Verification may not have processed (timeout or no matches)"
        fi
    else
        test_warning "Search with verification timed out or failed (may be expected)"
    fi
}

# Test: JSON Output Format
test_json_output() {
    test_start "JSON Output Format"
    
    local results="$RESULTS_DIR/search_json.txt"
    if $BINARY_DIR/rs-bgsearch -a "config" -d "$INDEX_DIR" -o json > "$results" 2>&1; then
        if python3 -m json.tool "$results" > /dev/null 2>&1; then
            test_pass "JSON output is valid"
        else
            test_fail "JSON output is invalid"
        fi
    else
        test_fail "JSON output generation failed"
    fi
}

# Test: File Parser
test_file_parser() {
    test_start "File Parser"
    
    # Parse JSON file
    local results="$RESULTS_DIR/parse_json.txt"
    if $BINARY_DIR/rs-bgparse --input "$DATA_DIR/users.json" --extract "users.*username" --format json > "$results" 2>&1; then
        if grep -q "admin" "$results" && grep -q "user1" "$results"; then
            test_pass "JSON parsing extracted expected data"
        else
            test_fail "JSON parsing did not extract expected data"
        fi
    else
        test_fail "JSON parsing failed"
    fi
    
    # Test CSV parsing
    echo "name,email,api_key" > "$DATA_DIR/test.csv"
    echo "admin,admin@example.com,key123" >> "$DATA_DIR/test.csv"
    echo "user,user@example.com,key456" >> "$DATA_DIR/test.csv"
    
    local csv_results="$RESULTS_DIR/parse_csv.txt"
    if $BINARY_DIR/rs-bgparse --input "$DATA_DIR/test.csv" --extract "name,email" --format csv > "$csv_results" 2>&1; then
        if grep -q "admin" "$csv_results"; then
            test_pass "CSV parsing works correctly"
        else
            test_fail "CSV parsing did not extract expected data"
        fi
    else
        test_fail "CSV parsing failed"
    fi
}

# Test: Batch Processing
test_batch_processing() {
    test_start "Batch Processing"
    
    # Create batch directory
    mkdir -p "$DATA_DIR/batch_test"
    cp "$DATA_DIR/config.txt" "$DATA_DIR/batch_test/"
    cp "$DATA_DIR/logs.txt" "$DATA_DIR/batch_test/"
    
    local results="$RESULTS_DIR/batch_parse.txt"
    if $BINARY_DIR/rs-bgparse batch --input "$DATA_DIR/batch_test" --format auto --output "$RESULTS_DIR/batch_output" > "$results" 2>&1; then
        if [ -d "$RESULTS_DIR/batch_output" ]; then
            test_pass "Batch processing completed"
        else
            test_fail "Batch processing did not create output directory"
        fi
    else
        test_fail "Batch processing failed"
    fi
}

# Test: File Verification
test_file_verification() {
    test_start "File Verification"
    
    # Verify patterns in files
    local results="$RESULTS_DIR/verify.txt"
    if $BINARY_DIR/rs-bgverify --input "$DATA_DIR/config.txt" --patterns "password" "api_key" > "$results" 2>&1; then
        if grep -q "password\|api_key" "$results"; then
            test_pass "File verification found expected patterns"
        else
            test_fail "File verification did not find expected patterns"
        fi
    else
        test_fail "File verification failed"
    fi
}

# Test: Archive Extraction
test_archive_extraction() {
    test_start "Archive Extraction"
    
    # Create a simple archive
    cd "$DATA_DIR"
    zip -q test_archive.zip config.txt logs.txt
    cd - > /dev/null
    
    local extract_dir="$RESULTS_DIR/extracted"
    mkdir -p "$extract_dir"
    
    if $BINARY_DIR/rs-bgextractfile zip --input "$DATA_DIR/test_archive.zip" --output "$extract_dir" > "$RESULTS_DIR/extract.log" 2>&1; then
        if [ -f "$extract_dir/config.txt" ] && [ -f "$extract_dir/logs.txt" ]; then
            test_pass "Archive extraction successful"
        else
            test_fail "Archive extraction did not extract expected files"
        fi
    else
        test_fail "Archive extraction failed"
    fi
    
    # Test listing archive contents
    local list_results="$RESULTS_DIR/list_archive.txt"
    if $BINARY_DIR/rs-bgextractfile list --input "$DATA_DIR/test_archive.zip" > "$list_results" 2>&1; then
        if grep -q "config.txt" "$list_results" && grep -q "logs.txt" "$list_results"; then
            test_pass "Archive listing works correctly"
        else
            test_fail "Archive listing did not show expected files"
        fi
    else
        test_fail "Archive listing failed"
    fi
}

# Test: Pattern-Based Extraction
test_pattern_extraction() {
    test_start "Pattern-Based Extraction"
    
    local extract_dir="$RESULTS_DIR/pattern_extracted"
    mkdir -p "$extract_dir"
    
    if $BINARY_DIR/rs-bgextractfile pattern --input "$DATA_DIR" \
        --output "$extract_dir" \
        "config.txt" "*.log" > "$RESULTS_DIR/pattern_extract.log" 2>&1; then
        local extracted_count=$(find "$extract_dir" -type f | wc -l)
        if [ "$extracted_count" -gt 0 ]; then
            test_pass "Pattern-based extraction worked ($extracted_count files)"
        else
            test_fail "Pattern-based extraction did not extract files"
        fi
    else
        test_fail "Pattern-based extraction failed"
    fi
}

# Test: Performance with Large Dataset
test_performance() {
    test_start "Performance Test with Large Dataset"
    
    # Use existing large file
    if [ -f "$DATA_DIR/large.txt" ]; then
        # Add to file list and reindex
        find "$DATA_DIR" -name "large.txt" -exec echo {} \; > "$DATA_DIR/large_list.txt"
        $BINARY_DIR/rs-bgindex -p "$INDEX_DIR/performance" < "$DATA_DIR/large_list.txt" > /dev/null 2>&1
        
        local start_time=$(date +%s.%N)
        if $BINARY_DIR/rs-bgsearch -a "password" -d "$INDEX_DIR" > "$RESULTS_DIR/perf_search.txt" 2>&1; then
            local end_time=$(date +%s.%N)
            local duration=$(echo "$end_time - $start_time" | bc -l 2>/dev/null || echo "N/A")
            
            local result_count=$(grep -c "password" "$RESULTS_DIR/perf_search.txt")
            test_pass "Performance test completed in ${duration}s with $result_count results"
        else
            test_fail "Performance test search failed"
        fi
    else
        test_warning "Large dataset not available for performance test"
    fi
}

# Test: Error Handling
test_error_handling() {
    test_start "Error Handling"
    
    # Test with non-existent index
    if ! $BINARY_DIR/rs-bgsearch -a "test" -d "/nonexistent" > "$RESULTS_DIR/error_test.txt" 2>&1; then
        test_pass "Correctly handles non-existent index directory"
    else
        test_fail "Should fail with non-existent index directory"
    fi
    
    # Test with invalid pattern
    if $BINARY_DIR/rs-bgsearch -a "" -d "$INDEX_DIR" > "$RESULTS_DIR/empty_pattern.txt" 2>&1; then
        test_pass "Handles empty pattern gracefully"
    else
        test_warning "Empty pattern handling may need improvement"
    fi
    
    # Test with non-existent file
    if ! $BINARY_DIR/rs-bgparse --input "/nonexistent/file.txt" > /dev/null 2>&1; then
        test_pass "Correctly handles non-existent input file"
    else
        test_fail "Should fail with non-existent input file"
    fi
}

# Test: Index Verification
test_index_verification() {
    test_start "Index Verification"
    
    if [ -f "$INDEX_DIR/test_index.bgi" ]; then
        local results="$RESULTS_DIR/index_verify.txt"
        if $BINARY_DIR/rs-bgverify index --index "$INDEX_DIR/test_index.bgi" > "$results" 2>&1; then
            if grep -q "valid\|consistent" "$results" || [ -s "$results" ]; then
                test_pass "Index verification completed"
            else
                test_warning "Index verification completed but output unclear"
            fi
        else
            test_fail "Index verification failed"
        fi
    else
        test_warning "No index file available for verification test"
    fi
}

# Test: Concurrent Operations
test_concurrent_operations() {
    test_start "Concurrent Operations"
    
    # Run multiple searches in parallel
    local pids=()
    for pattern in "password" "api_key" "config" "error"; do
        $BINARY_DIR/rs-bgsearch -a "$pattern" -d "$INDEX_DIR" > "$RESULTS_DIR/concurrent_$pattern.txt" 2>&1 &
        pids+=($!)
    done
    
    # Wait for all searches to complete
    local failed=0
    for pid in "${pids[@]}"; do
        if ! wait "$pid"; then
            failed=$((failed + 1))
        fi
    done
    
    if [ "$failed" -eq 0 ]; then
        test_pass "Concurrent operations completed successfully"
    else
        test_fail "$failed concurrent operations failed"
    fi
}

# Test: End-to-End Workflow
test_end_to_end() {
    test_start "End-to-End Workflow"
    
    # Create fresh index
    local e2e_index="$INDEX_DIR/e2e_test"
    local e2e_results="$RESULTS_DIR/e2e_search.txt"
    
    # 1. Index files
    if ! $BINARY_DIR/rs-bgindex -p "$e2e_index" < "$DATA_DIR/file_list.txt" > /dev/null 2>&1; then
        test_fail "End-to-end: Index building failed"
        return
    fi
    
    # 2. Search for patterns
    if ! $BINARY_DIR/rs-bgsearch -a "password" -d "$INDEX_DIR" > "$e2e_results" 2>&1; then
        test_fail "End-to-end: Search failed"
        return
    fi
    
    # 3. Verify results
    local verify_results="$RESULTS_DIR/e2e_verify.txt"
    if ! $BINARY_DIR/rs-bgverify --input "$DATA_DIR/config.txt" --patterns "password" > "$verify_results" 2>&1; then
        test_fail "End-to-end: Verification failed"
        return
    fi
    
    # 4. Extract files
    local extract_dir="$RESULTS_DIR/e2e_extracted"
    mkdir -p "$extract_dir"
    if ! $BINARY_DIR/rs-bgextractfile pattern --input "$DATA_DIR" --output "$extract_dir" "config.txt" > /dev/null 2>&1; then
        test_fail "End-to-end: File extraction failed"
        return
    fi
    
    # 5. Parse extracted files
    local parse_results="$RESULTS_DIR/e2e_parse.txt"
    if ! $BINARY_DIR/rs-bgparse --input "$extract_dir/config.txt" --extract "api_key" --format text > "$parse_results" 2>&1; then
        test_fail "End-to-end: File parsing failed"
        return
    fi
    
    # Verify all steps produced expected output
    local success=true
    [ -s "$e2e_results" ] || success=false
    [ -s "$verify_results" ] || success=false
    [ -d "$extract_dir" ] && [ $(find "$extract_dir" -type f | wc -l) -gt 0 ] || success=false
    [ -s "$parse_results" ] || success=false
    
    if [ "$success" = true ]; then
        test_pass "End-to-end workflow completed successfully"
    else
        test_fail "End-to-end workflow incomplete (some steps failed)"
    fi
}

# Generate test report
generate_test_report() {
    local report_file="$TEST_DIR/test_report.txt"
    
    cat > "$report_file" << EOF
BigGrep Rust Integration Test Report
====================================

Test Run Summary:
- Total Tests: $TESTS_RUN
- Passed: $TESTS_PASSED
- Failed: $TESTS_FAILED
- Success Rate: $(echo "scale=1; $TESTS_PASSED * 100 / $TESTS_RUN" | bc 2>/dev/null || echo "N/A")%

Test Environment:
- Test Directory: $TEST_DIR
- Data Directory: $DATA_DIR
- Index Directory: $INDEX_DIR
- Results Directory: $RESULTS_DIR
- Binary Directory: $BINARY_DIR

Generated: $(date)
EOF
    
    print_status "Test report saved to: $report_file"
}

# Main test runner
run_all_tests() {
    print_status "Starting BigGrep Rust Integration Test Suite"
    echo "=============================================="
    
    # Setup
    setup_test_environment
    create_test_data
    
    # Run tests
    test_basic_indexing
    test_basic_search
    test_binary_search
    test_unicode_search
    test_metadata_filtering
    test_search_with_verification
    test_json_output
    test_file_parser
    test_batch_processing
    test_file_verification
    test_archive_extraction
    test_pattern_extraction
    test_performance
    test_error_handling
    test_index_verification
    test_concurrent_operations
    test_end_to_end
    
    # Generate report
    generate_test_report
    
    # Final summary
    echo ""
    print_status "Test Suite Completed"
    echo "====================="
    print_status "Total Tests: $TESTS_RUN"
    print_success "Passed: $TESTS_PASSED"
    if [ $TESTS_FAILED -gt 0 ]; then
        print_failure "Failed: $TESTS_FAILED"
    else
        print_success "Failed: $TESTS_FAILED"
    fi
    
    if [ $TESTS_FAILED -eq 0 ]; then
        print_success "All tests passed!"
        exit 0
    else
        print_failure "Some tests failed. Check results for details."
        exit 1
    fi
}

# Cleanup function
cleanup() {
    if [ "${KEEP_TEST_DIR:-false}" != "true" ]; then
        print_status "Cleaning up test environment..."
        rm -rf "$TEST_DIR"
    fi
}

# Set trap for cleanup
trap cleanup EXIT

# Check for help flag
if [ "$1" = "--help" ] || [ "$1" = "-h" ]; then
    echo "BigGrep Rust Integration Test Suite"
    echo ""
    echo "Usage: $0 [OPTIONS]"
    echo ""
    echo "Options:"
    echo "  --keep-test-dir    Don't delete test directory after completion"
    echo "  -h, --help         Show this help message"
    echo ""
    echo "This test suite validates all BigGrep Rust tools working together"
    echo "in realistic scenarios including:"
    echo "  - Index building and search"
    echo "  - Various search patterns (ASCII, binary, Unicode)"
    echo "  - File parsing and verification"
    echo "  - Archive extraction"
    echo "  - Performance testing"
    echo "  - Error handling"
    echo "  - End-to-end workflows"
    echo ""
    exit 0
fi

# Run all tests
run_all_tests