//! BigGrep Verifier (rs-bgverify)
//!
//! Verifies search results using Boyer-Moore-Horspool fast string search algorithm
//! with multi-pattern verification (AND logic), memory-mapped file access,
//! and 256-character skip table optimization.

use clap::Parser;
use std::path::{Path, PathBuf};
use anyhow::{Result, Context};
use log::{info, warn, error, debug};
use biggrep_core::search::boyer_moore::{BoyerMooreHorspool, utils};
use rayon::prelude::*;

/// Boyer-Moore-Horspool Verifier CLI
#[derive(Parser)]
#[command(name = "rs-bgverify")]
#[command(about = "BigGrep verifier using Boyer-Moore-Horspool algorithm")]
#[command(version = "0.1.0")]
struct Cli {
    /// Patterns to search for (comma-separated or multiple -p flags)
    #[arg(short = 'p', long, required = true, value_delimiter = ',', 
          help = "Search patterns to verify (case-sensitive)")]
    patterns: Vec<String>,
    
    /// Files to verify
    #[arg(short = 'f', long, required = true,
          help = "Files to verify against patterns")]
    files: Vec<PathBuf>,
    
    /// Binary mode - search in binary files
    #[arg(short = 'b', long,
          help = "Enable binary mode for hex pattern search")]
    binary: bool,
    
    /// Verbose output
    #[arg(short = 'v', long,
          help = "Enable verbose output")]
    verbose: bool,
    
    /// Case-sensitive search (default: case-insensitive)
    #[arg(short = 'c', long,
          help = "Enable case-sensitive search")]
    case_sensitive: bool,
    
    /// Output results to file
    #[arg(short = 'o', long,
          help = "Output results to file")]
    output: Option<PathBuf>,
    
    /// Number of parallel threads (default: CPU cores)
    #[arg(short = 't', long,
          help = "Number of parallel threads")]
    threads: Option<usize>,
}

/// Verification result for a single file
#[derive(Debug, Clone)]
struct FileVerification {
    file_path: PathBuf,
    file_size: u64,
    all_patterns_found: bool,
    pattern_positions: Vec<(usize, usize)>,
    search_successful: bool,
}

impl FileVerification {
    fn success(file_path: PathBuf, file_size: u64) -> Self {
        Self {
            file_path,
            file_size,
            all_patterns_found: true,
            pattern_positions: Vec::new(),
            search_successful: true,
        }
    }
    
    fn failure(file_path: PathBuf, file_size: u64) -> Self {
        Self {
            file_path,
            file_size,
            all_patterns_found: false,
            pattern_positions: Vec::new(),
            search_successful: false,
        }
    }
}

/// Main function
fn main() -> Result<()> {
    let cli = Cli::parse();
    
    // Initialize logging
    init_logging(cli.verbose);
    
    info!("BigGrep Boyer-Moore-Horspool Verifier v0.1.0 starting");
    info!("Using Boyer-Moore-Horspool algorithm with 256-character skip table");
    info!("Multi-pattern verification (AND logic): {} patterns", cli.patterns.len());
    info!("Verifying {} file(s)", cli.files.len());
    
    if cli.verbose {
        info!("Search mode: {}", if cli.case_sensitive { "Case-sensitive" } else { "Case-insensitive" });
        info!("Binary mode: {}", if cli.binary { "Enabled" } else { "Disabled" });
        if let Some(threads) = cli.threads {
            info!("Parallel threads: {}", threads);
        } else {
            info!("Parallel threads: {} (CPU cores)", rayon::current_num_threads());
        }
    }
    
    // Parse and prepare patterns
    let patterns = parse_patterns(&cli.patterns, cli.binary)?;
    info!("Prepared {} patterns for verification", patterns.len());
    
    // Set thread count if specified
    let thread_count = cli.threads.unwrap_or_else(|| rayon::current_num_threads());
    if thread_count != rayon::current_num_threads() {
        rayon::ThreadPoolBuilder::new()
            .num_threads(thread_count)
            .build_global()
            .map_err(|_| anyhow::anyhow!("Failed to set thread pool size"))?;
    }
    
    // Verify files
    let results = verify_files_parallel(&cli.files, &patterns, cli.case_sensitive, cli.binary)?;
    
    // Output results
    if let Some(output_file) = &cli.output {
        output_results_to_file(&results, output_file)?;
    } else {
        output_results_to_stdout(&results, cli.verbose)?;
    }
    
    // Summary
    let successful_files = results.iter().filter(|r| r.search_successful).count();
    let total_files = results.len();
    
    println!("\nVerification Summary:");
    println!("  Total files: {}", total_files);
    println!("  Successful: {}", successful_files);
    println!("  Failed: {}", total_files - successful_files);
    println!("  Success rate: {:.1}%", (successful_files as f64 / total_files as f64) * 100.0);
    
    Ok(())
}

/// Initialize logging based on verbosity level
fn init_logging(verbose: bool) {
    let log_level = if verbose {
        log::LevelFilter::Debug
    } else {
        log::LevelFilter::Info
    };
    
    env_logger::Builder::from_default_env()
        .filter_level(log_level)
        .init();
}

/// Parse patterns from CLI input
fn parse_patterns(pattern_strings: &[String], binary_mode: bool) -> Result<Vec<Vec<u8>>> {
    let mut patterns = Vec::new();
    
    for pattern_str in pattern_strings {
        let pattern = if binary_mode {
            // Parse hex pattern (e.g., "DEADBEEF", "DE AD BE EF")
            parse_hex_pattern(pattern_str)?
        } else {
            // Treat as ASCII string
            pattern_str.as_bytes().to_vec()
        };
        
        if !pattern.is_empty() {
            patterns.push(pattern);
        } else {
            warn!("Skipping empty pattern: {}", pattern_str);
        }
    }
    
    if patterns.is_empty() {
        anyhow::bail!("No valid patterns found");
    }
    
    Ok(patterns)
}

/// Parse hexadecimal pattern string
fn parse_hex_pattern(hex_str: &str) -> Result<Vec<u8>> {
    let hex_cleaned = hex_str.replace(" ", "").replace("0x", "");
    
    if hex_cleaned.len() % 2 != 0 {
        anyhow::bail!("Invalid hex pattern length: {}", hex_str);
    }
    
    let mut bytes = Vec::new();
    for i in (0..hex_cleaned.len()).step_by(2) {
        let hex_byte = &hex_cleaned[i..i+2];
        let byte = u8::from_str_radix(hex_byte, 16)
            .map_err(|_| anyhow::anyhow!("Invalid hex byte: {}", hex_byte))?;
        bytes.push(byte);
    }
    
    Ok(bytes)
}

/// Verify files in parallel using Boyer-Moore-Horspool
fn verify_files_parallel(
    files: &[PathBuf],
    patterns: &[Vec<u8>],
    case_sensitive: bool,
    binary_mode: bool,
) -> Result<Vec<FileVerification>> {
    info!("Starting parallel verification of {} files", files.len());
    
    let results: Vec<FileVerification> = files
        .par_iter()
        .filter_map(|file| {
            match verify_single_file(file, patterns, case_sensitive, binary_mode) {
                Ok(result) => Some(result),
                Err(e) => {
                    warn!("Error verifying file {:?}: {}", file, e);
                    Some(FileVerification::failure(file.clone(), 0))
                }
            }
        })
        .collect();
    
    Ok(results)
}

/// Verify a single file using Boyer-Moore-Horspool
fn verify_single_file(
    file_path: &PathBuf,
    patterns: &[Vec<u8>],
    case_sensitive: bool,
    binary_mode: bool,
) -> Result<FileVerification> {
    // Check if file exists and get metadata
    let metadata = std::fs::metadata(file_path)
        .with_context(|| format!("Failed to get metadata for file: {:?}", file_path))?;
    
    let file_size = metadata.len();
    
    if file_size == 0 {
        warn!("Skipping empty file: {:?}", file_path);
        return Ok(FileVerification::failure(file_path.clone(), file_size));
    }
    
    if file_size > 1024 * 1024 * 1024 { // 1GB limit
        warn!("Skipping large file ({} bytes): {:?}", file_size, file_path);
        return Ok(FileVerification::failure(file_path.clone(), file_size));
    }
    
    debug!("Verifying file: {:?} ({} bytes)", file_path, file_size);
    
    // Use Boyer-Moore-Horspool verification
    let verification_result = utils::verify_patterns_in_file(
        file_path,
        patterns,
        case_sensitive,
        binary_mode
    ).with_context(|| format!("Failed to verify file: {:?}", file_path))?;
    
    let file_verification = FileVerification {
        file_path: file_path.clone(),
        file_size,
        all_patterns_found: verification_result.all_patterns_found,
        pattern_positions: verification_result.pattern_positions,
        search_successful: verification_result.verification_successful,
    };
    
    debug!("File verification result: success={}, patterns_found={}/{}", 
           file_verification.search_successful, 
           file_verification.pattern_positions.len(),
           patterns.len());
    
    Ok(file_verification)
}

/// Output results to stdout
fn output_results_to_stdout(results: &[FileVerification], verbose: bool) -> Result<()> {
    println!("\nVerification Results:");
    println!("{}", "=".repeat(80));
    
    let mut success_count = 0;
    let mut failure_count = 0;
    
    for result in results {
        let status = if result.search_successful { "✓ PASS" } else { "✗ FAIL" };
        let all_patterns = if result.all_patterns_found { "ALL" } else { "PARTIAL" };
        
        print!("{:8} {:8} {:10} bytes  ", status, all_patterns, result.file_size);
        println!("{}", result.file_path.display());
        
        if verbose && result.search_successful && !result.pattern_positions.is_empty() {
            println!("        Patterns found at positions:");
            for (pattern_idx, position) in &result.pattern_positions {
                println!("          Pattern {}: position {}", pattern_idx, position);
            }
        }
        
        if result.search_successful {
            success_count += 1;
        } else {
            failure_count += 1;
        }
    }
    
    println!("\nSummary: {} passed, {} failed", success_count, failure_count);
    
    Ok(())
}

/// Output results to file
fn output_results_to_file(results: &[FileVerification], output_file: &PathBuf) -> Result<()> {
    let json = serde_json::to_string_pretty(results)?;
    std::fs::write(output_file, json)?;
    info!("Results saved to: {:?}", output_file);
    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;
    
    #[test]
    fn test_parse_patterns_ascii() {
        let patterns = vec!["hello".to_string(), "world".to_string()];
        let parsed = parse_patterns(&patterns, false).unwrap();
        assert_eq!(parsed, vec![b"hello".to_vec(), b"world".to_vec()]);
    }
    
    #[test]
    fn test_parse_patterns_hex() {
        let patterns = vec!["DEADBEEF".to_string()];
        let parsed = parse_patterns(&patterns, true).unwrap();
        assert_eq!(parsed, vec![vec![0xDE, 0xAD, 0xBE, 0xEF]]);
    }
    
    #[test]
    fn test_parse_hex_pattern() {
        let hex_str = "DE AD BE EF";
        let parsed = parse_hex_pattern(hex_str).unwrap();
        assert_eq!(parsed, vec![0xDE, 0xAD, 0xBE, 0xEF]);
    }
}