#!/bin/bash
# rs-bgsearch Integration Test Script
# 
# This script demonstrates basic functionality of rs-bgsearch

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo "============================================"
echo "rs-bgsearch Integration Tests"
echo "============================================"

# Check if binary exists
if [ ! -f "target/release/rs-bgsearch" ] && [ ! -f "target/debug/rs-bgsearch" ]; then
    echo -e "${RED}ERROR: rs-bgsearch binary not found${NC}"
    echo "Please build the project first: cargo build"
    exit 1
fi

# Use debug build if release not available
if [ -f "target/debug/rs-bgsearch" ]; then
    RS_BGSEARCH="./target/debug/rs-bgsearch"
else
    RS_BGSEARCH="./target/release/rs-bgsearch"
fi

echo -e "${GREEN}Using rs-bgsearch: $RS_BGSEARCH${NC}"
echo ""

# Test 1: Help message
echo -e "${YELLOW}Test 1: Help message${NC}"
if $RS_BGSEARCH --help > /dev/null 2>&1; then
    echo -e "${GREEN}✓ Help message works${NC}"
else
    echo -e "${RED}✗ Help message failed${NC}"
fi
echo ""

# Test 2: Version information
echo -e "${YELLOW}Test 2: Version information${NC}"
if $RS_BGSEARCH --version > /dev/null 2>&1; then
    echo -e "${GREEN}✓ Version information works${NC}"
else
    echo -e "${RED}✗ Version information failed${NC}"
fi
echo ""

# Test 3: No arguments (should show usage)
echo -e "${YELLOW}Test 3: No arguments (should show usage)${NC}"
if $RS_BGSEARCH 2>&1 | grep -q "error: .*required"; then
    echo -e "${GREEN}✓ Properly rejects missing arguments${NC}"
else
    echo -e "${RED}✗ Should reject missing arguments${NC}"
fi
echo ""

# Test 4: Create test index directory
echo -e "${YELLOW}Test 4: Create test data${NC}"
mkdir -p test_indexes

# Create a simple test index file (mock)
echo "BGI1" > test_indexes/test.bgi
echo "3" >> test_indexes/test.bgi
dd if=/dev/zero bs=1024 count=10 >> test_indexes/test.bgi 2>/dev/null

if [ -f "test_indexes/test.bgi" ]; then
    echo -e "${GREEN}✓ Test data created${NC}"
else
    echo -e "${RED}✗ Failed to create test data${NC}"
fi
echo ""

# Test 5: Search with pattern but no indexes (should handle gracefully)
echo -e "${YELLOW}Test 5: Search with non-existent indexes${NC}"
if $RS_BGSEARCH -a "test" -d /nonexistent 2>&1 | grep -q "No index files"; then
    echo -e "${GREEN}✓ Gracefully handles missing indexes${NC}"
else
    echo -e "${YELLOW}⚠ May not properly handle missing indexes${NC}"
fi
echo ""

# Test 6: Pattern type detection
echo -e "${YELLOW}Test 6: Pattern type detection (ASCII)${NC}"
if $RS_BGSEARCH -a "hello world" --help > /dev/null 2>&1; then
    echo -e "${GREEN}✓ ASCII pattern accepted${NC}"
else
    echo -e "${RED}✗ ASCII pattern failed${NC}"
fi

echo -e "${YELLOW}Test 7: Pattern type detection (Hex)${NC}"
if $RS_BGSEARCH -b "68656c6c6f" --help > /dev/null 2>&1; then
    echo -e "${GREEN}✓ Hex pattern accepted${NC}"
else
    echo -e "${RED}✗ Hex pattern failed${NC}"
fi

echo -e "${YELLOW}Test 8: Pattern type detection (Unicode)${NC}"
if $RS_BGSEARCH -u "café" --help > /dev/null 2>&1; then
    echo -e "${GREEN}✓ Unicode pattern accepted${NC}"
else
    echo -e "${RED}✗ Unicode pattern failed${NC}"
fi
echo ""

# Test 9: Filter parsing
echo -e "${YELLOW}Test 9: Metadata filter parsing${NC}"
if $RS_BGSEARCH -a "test" -f "size>=1024" --help > /dev/null 2>&1; then
    echo -e "${GREEN}✓ Filter parsing works${NC}"
else
    echo -e "${RED}✗ Filter parsing failed${NC}"
fi
echo ""

# Test 10: Output format options
echo -e "${YELLOW}Test 10: Output format options${NC}"
for format in text json csv; do
    if $RS_BGSEARCH -a "test" -o $format --help > /dev/null 2>&1; then
        echo -e "${GREEN}✓ Output format '$format' supported${NC}"
    else
        echo -e "${RED}✗ Output format '$format' failed${NC}"
    fi
done
echo ""

# Test 11: Configuration file loading
echo -e "${YELLOW}Test 11: Configuration file validation${NC}"
if [ -f "bgsearch.conf.example" ]; then
    echo -e "${GREEN}✓ Example configuration file exists${NC}"
else
    echo -e "${RED}✗ Example configuration file missing${NC}"
fi
echo ""

# Cleanup
echo -e "${YELLOW}Cleaning up test data${NC}"
rm -rf test_indexes
echo -e "${GREEN}✓ Cleanup complete${NC}"
echo ""

echo "============================================"
echo "Integration tests completed!"
echo "============================================"
echo ""
echo "To run full functionality tests:"
echo "1. Build actual BigGrep indexes using bgindex"
echo "2. Run: $RS_BGSEARCH -a 'pattern' -d /path/to/indexes"
echo ""
echo "For YARA support:"
echo "1. Build with feature: cargo build --features yara"
echo "2. Run: $RS_BGSEARCH -a 'pattern' -y rules.yar"
