/// Pattern module - handles ASCII, binary, and Unicode pattern conversion
use anyhow::{Result, Context};
use std::fmt;

/// Supported pattern types
#[derive(Debug, Clone, PartialEq, Eq)]
pub enum PatternType {
    Ascii,
    Binary,
    Unicode,
}

impl fmt::Display for PatternType {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            PatternType::Ascii => write!(f, "ASCII"),
            PatternType::Binary => write!(f, "Binary/Hex"),
            PatternType::Unicode => write!(f, "Unicode"),
        }
    }
}

/// Search pattern with conversion capabilities
#[derive(Debug, Clone)]
pub struct Pattern {
    pub pattern_type: PatternType,
    pub bytes: Vec<u8>,
    pub ngrams: Option<Vec<Vec<u8>>>,
}

impl Pattern {
    /// Create pattern from ASCII string
    pub fn from_ascii(text: &str) -> Result<Self> {
        let bytes = text.as_bytes().to_vec();
        
        Ok(Self {
            pattern_type: PatternType::Ascii,
            bytes,
            ngrams: None,
        })
    }

    /// Create pattern from hexadecimal string
    pub fn from_hex(hex: &str) -> Result<Self> {
        let cleaned_hex = hex.replace("0x", "").replace(" ", "").replace(":", "");
        let bytes = hex::decode(&cleaned_hex)
            .with_context(|| format!("Invalid hex string: {}", hex))?;
        
        Ok(Self {
            pattern_type: PatternType::Binary,
            bytes,
            ngrams: None,
        })
    }

    /// Create pattern from Unicode string
    pub fn from_unicode(text: &str) -> Result<Self> {
        // Simple UTF-8 encoding for now
        // In a full implementation, would handle proper Unicode normalization
        let bytes = text.as_bytes().to_vec();
        
        Ok(Self {
            pattern_type: PatternType::Unicode,
            bytes,
            ngrams: None,
        })
    }

    /// Auto-detect pattern type and create pattern
    pub fn auto_detect(pattern: &str) -> Result<Self> {
        // Try ASCII first
        if pattern.chars().all(|c| c.is_ascii()) {
            return Self::from_ascii(pattern);
        }

        // Try Unicode
        if pattern.chars().any(|c| c as u32 > 127) {
            return Self::from_unicode(pattern);
        }

        // Try hex format
        if pattern.starts_with("0x") || 
           pattern.replace(" ", "").chars().all(|c| c.is_ascii_hexdigit()) {
            return Self::from_hex(pattern);
        }

        // Default to ASCII
        Self::from_ascii(pattern)
    }

    /// Convert pattern to N-grams of specified size
    pub fn convert_to_ngrams(&mut self, n: usize) -> Result<()> {
        if self.bytes.len() < n {
            return Ok(()); // Not enough bytes for N-grams
        }

        let mut ngrams = Vec::new();
        for i in 0..=self.bytes.len() - n {
            let ngram = self.bytes[i..i + n].to_vec();
            ngrams.push(ngram);
        }

        self.ngrams = Some(ngrams);
        Ok(())
    }

    /// Get N-grams if available
    pub fn get_ngrams(&self) -> &[Vec<u8>] {
        self.ngrams.as_ref().map(|v| v.as_slice()).unwrap_or(&[])
    }

    /// Get pattern as hex string for display
    pub fn to_hex_string(&self) -> String {
        self.bytes.iter().map(|b| format!("{:02x}", b)).collect()
    }

    /// Get pattern as ASCII string for display (non-printable chars replaced)
    pub fn to_ascii_string(&self) -> String {
        self.bytes.iter()
            .map(|&b| if b.is_ascii_graphic() || b == b' ' { b as char } else { '.' })
            .collect()
    }
}

/// Term conversion utilities
pub struct TermConverter;

impl TermConverter {
    /// Convert various term formats to normalized hex
    pub fn normalize_to_hex(pattern: &Pattern) -> Result<String> {
        Ok(pattern.to_hex_string())
    }

    /// Validate that pattern can be searched
    pub fn validate_for_search(pattern: &Pattern) -> Result<()> {
        if pattern.bytes.is_empty() {
            anyhow::bail!("Empty pattern");
        }

        if pattern.bytes.len() > 1024 {
            warn!("Pattern is very long ({} bytes), may be slow", pattern.bytes.len());
        }

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_ascii_pattern() {
        let pattern = Pattern::from_ascii("hello").unwrap();
        assert_eq!(pattern.pattern_type, PatternType::Ascii);
        assert_eq!(pattern.bytes, b"hello".to_vec());
    }

    #[test]
    fn test_hex_pattern() {
        let pattern = Pattern::from_hex("68656c6c6f").unwrap();
        assert_eq!(pattern.pattern_type, PatternType::Binary);
        assert_eq!(pattern.bytes, b"hello".to_vec());
    }

    #[test]
    fn test_hex_pattern_with_prefix() {
        let pattern = Pattern::from_hex("0x68656c6c6f").unwrap();
        assert_eq!(pattern.pattern_type, PatternType::Binary);
        assert_eq!(pattern.bytes, b"hello".to_vec());
    }

    #[test]
    fn test_auto_detect_ascii() {
        let pattern = Pattern::auto_detect("hello").unwrap();
        assert_eq!(pattern.pattern_type, PatternType::Ascii);
    }

    #[test]
    fn test_auto_detect_hex() {
        let pattern = Pattern::auto_detect("0x68656c6c6f").unwrap();
        assert_eq!(pattern.pattern_type, PatternType::Binary);
    }

    #[test]
    fn test_ngram_conversion() {
        let mut pattern = Pattern::from_ascii("hello").unwrap();
        pattern.convert_to_ngrams(3).unwrap();
        
        let ngrams = pattern.get_ngrams();
        assert_eq!(ngrams.len(), 3); // "hel", "ell", "llo"
        assert_eq!(ngrams[0], b"hel".to_vec());
        assert_eq!(ngrams[1], b"ell".to_vec());
        assert_eq!(ngrams[2], b"llo".to_vec());
    }
}
