/// Metadata filtering module
use anyhow::{Result, Context};
use std::collections::HashMap;
use std::str::FromStr;

/// Filter comparison operators
#[derive(Debug, Clone, PartialEq, Eq)]
pub enum ComparisonOp {
    Equal,
    NotEqual,
    Less,
    Greater,
    LessEqual,
    GreaterEqual,
}

impl ComparisonOp {
    pub fn from_str(op: &str) -> Result<Self> {
        match op {
            "=" => Ok(ComparisonOp::Equal),
            "!=" => Ok(ComparisonOp::NotEqual),
            "<" => Ok(ComparisonOp::Less),
            ">" => Ok(ComparisonOp::Greater),
            "<=" => Ok(ComparisonOp::LessEqual),
            ">=" => Ok(ComparisonOp::GreaterEqual),
            _ => anyhow::bail!("Invalid comparison operator: {}", op),
        }
    }
}

/// Metadata filter expression
#[derive(Debug, Clone)]
pub struct MetadataFilter {
    pub field: String,
    pub operator: ComparisonOp,
    pub value: String,
}

impl MetadataFilter {
    /// Parse a filter expression (e.g., "size>=1024", "arch=x86_64")
    pub fn parse(filter_str: &str) -> Result<Self> {
        let trimmed = filter_str.trim();
        
        // Try to find operators in order of preference
        let operators = ["!=", ">=", "<=", "=", "<", ">"];
        let mut found_op = None;
        let mut found_pos = 0;
        
        for op in &operators {
            if let Some(pos) = trimmed.find(op) {
                found_op = Some(op);
                found_pos = pos;
                break;
            }
        }
        
        if found_op.is_none() {
            anyhow::bail!("No valid operator found in filter: {}", filter_str);
        }
        
        let op_str = found_op.unwrap();
        let left = &trimmed[..found_pos];
        let right = &trimmed[found_pos + op_str.len()..];
        
        if left.trim().is_empty() {
            anyhow::bail!("Empty field name in filter: {}", filter_str);
        }
        
        if right.trim().is_empty() {
            anyhow::bail!("Empty value in filter: {}", filter_str);
        }
        
        Ok(Self {
            field: left.trim().to_string(),
            operator: ComparisonOp::from_str(op_str)?,
            value: right.trim().to_string(),
        })
    }

    /// Evaluate filter against metadata
    pub fn evaluate(&self, metadata: &HashMap<String, String>) -> Result<bool, FilterError> {
        let field_value = metadata.get(&self.field)
            .ok_or_else(|| FilterError::MissingKey(self.field.clone()))?;
        
        // Try numeric comparison first
        if let (Ok(metadata_num), Ok(filter_num)) = 
            (field_value.parse::<i64>(), self.value.parse::<i64>()) {
            
            match self.operator {
                ComparisonOp::Equal => Ok(metadata_num == filter_num),
                ComparisonOp::NotEqual => Ok(metadata_num != filter_num),
                ComparisonOp::Less => Ok(metadata_num < filter_num),
                ComparisonOp::Greater => Ok(metadata_num > filter_num),
                ComparisonOp::LessEqual => Ok(metadata_num <= filter_num),
                ComparisonOp::GreaterEqual => Ok(metadata_num >= filter_num),
            }
        } else {
            // String comparison
            match self.operator {
                ComparisonOp::Equal => Ok(field_value == &self.value),
                ComparisonOp::NotEqual => Ok(field_value != &self.value),
                ComparisonOp::Less => Ok(field_value < &self.value),
                ComparisonOp::Greater => Ok(field_value > &self.value),
                ComparisonOp::LessEqual => Ok(field_value <= &self.value),
                ComparisonOp::GreaterEqual => Ok(field_value >= &self.value),
            }
        }
    }
}

/// Filter evaluation errors
#[derive(Debug, thiserror::Error)]
pub enum FilterError {
    #[error("Missing metadata field: {0}")]
    MissingKey(String),
    #[error("Invalid numeric value: {0}")]
    InvalidNumber(#[from] std::num::ParseIntError),
    #[error("Parse error: {0}")]
    ParseError(String),
}

/// Filter chain for applying multiple filters
#[derive(Debug, Clone)]
pub struct FilterChain {
    filters: Vec<MetadataFilter>,
}

impl FilterChain {
    /// Create a new filter chain
    pub fn new(filters: Vec<MetadataFilter>) -> Self {
        Self { filters }
    }

    /// Apply all filters to metadata
    pub fn evaluate(&self, metadata: &HashMap<String, String>) -> Result<bool, FilterError> {
        for filter in &self.filters {
            if !filter.evaluate(metadata)? {
                return Ok(false);
            }
        }
        Ok(true)
    }

    /// Get number of filters
    pub fn len(&self) -> usize {
        self.filters.len()
    }

    /// Check if chain is empty
    pub fn is_empty(&self) -> bool {
        self.filters.is_empty()
    }
}

/// Built-in metadata field names
pub struct MetadataFields;

impl MetadataFields {
    pub const SIZE: &'static str = "size";
    pub const ARCH: &'static str = "arch";
    pub const OS: &'static str = "os";
    pub const FILE_TYPE: &'static str = "type";
    pub const MD5: &'static str = "md5";
    pub const SHA1: &'static str = "sha1";
    pub const SHA256: &'static str = "sha256";
    pub const TIMESTAMP: &'static str = "timestamp";
    pub const PATH: &'static str = "path";
    pub const FILENAME: &'static str = "filename";
    pub const EXTENSION: &'static str = "extension";
    pub const PERMISSIONS: &'static str = "permissions";
    pub const OWNER: &'static str = "owner";
    pub const GROUP: &'static str = "group";
}

/// Common filter expressions for convenience
pub struct CommonFilters;

impl CommonFilters {
    /// Create filter for files larger than size
    pub fn size_greater_than(size: i64) -> MetadataFilter {
        MetadataFilter {
            field: MetadataFields::SIZE.to_string(),
            operator: ComparisonOp::Greater,
            value: size.to_string(),
        }
    }

    /// Create filter for files smaller than size
    pub fn size_less_than(size: i64) -> MetadataFilter {
        MetadataFilter {
            field: MetadataFields::SIZE.to_string(),
            operator: ComparisonOp::Less,
            value: size.to_string(),
        }
    }

    /// Create filter for specific architecture
    pub fn architecture(arch: &str) -> MetadataFilter {
        MetadataFilter {
            field: MetadataFields::ARCH.to_string(),
            operator: ComparisonOp::Equal,
            value: arch.to_string(),
        }
    }

    /// Create filter for specific operating system
    pub fn operating_system(os: &str) -> MetadataFilter {
        MetadataFilter {
            field: MetadataFields::OS.to_string(),
            operator: ComparisonOp::Equal,
            value: os.to_string(),
        }
    }

    /// Create filter for specific file type
    pub fn file_type(file_type: &str) -> MetadataFilter {
        MetadataFilter {
            field: MetadataFields::FILE_TYPE.to_string(),
            operator: ComparisonOp::Equal,
            value: file_type.to_string(),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_parse_equal_filter() {
        let filter = MetadataFilter::parse("size=1024").unwrap();
        assert_eq!(filter.field, "size");
        assert_eq!(filter.operator, ComparisonOp::Equal);
        assert_eq!(filter.value, "1024");
    }

    #[test]
    fn test_parse_greater_equal_filter() {
        let filter = MetadataFilter::parse("size>=65536").unwrap();
        assert_eq!(filter.field, "size");
        assert_eq!(filter.operator, ComparisonOp::GreaterEqual);
        assert_eq!(filter.value, "65536");
    }

    #[test]
    fn test_parse_not_equal_filter() {
        let filter = MetadataFilter::parse("os!=Windows").unwrap();
        assert_eq!(filter.field, "os");
        assert_eq!(filter.operator, ComparisonOp::NotEqual);
        assert_eq!(filter.value, "Windows");
    }

    #[test]
    fn test_numeric_filter_evaluation() {
        let filter = MetadataFilter::parse("size>=1024").unwrap();
        
        let mut metadata = HashMap::new();
        metadata.insert("size".to_string(), "2048".to_string());
        
        let result = filter.evaluate(&metadata).unwrap();
        assert!(result);
        
        // Test smaller value
        metadata.insert("size".to_string(), "512".to_string());
        let result = filter.evaluate(&metadata).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_string_filter_evaluation() {
        let filter = MetadataFilter::parse("arch=x86_64").unwrap();
        
        let mut metadata = HashMap::new();
        metadata.insert("arch".to_string(), "x86_64".to_string());
        
        let result = filter.evaluate(&metadata).unwrap();
        assert!(result);
        
        // Test different value
        metadata.insert("arch".to_string(), "arm64".to_string());
        let result = filter.evaluate(&metadata).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_missing_field_error() {
        let filter = MetadataFilter::parse("size>=1024").unwrap();
        
        let metadata = HashMap::new(); // No "size" field
        
        let result = filter.evaluate(&metadata);
        assert!(matches!(result, Err(FilterError::MissingKey(_))));
    }

    #[test]
    fn test_filter_chain() {
        let filters = vec![
            MetadataFilter::parse("size>=1024").unwrap(),
            MetadataFilter::parse("arch=x86_64").unwrap(),
        ];
        
        let chain = FilterChain::new(filters);
        
        let mut metadata = HashMap::new();
        metadata.insert("size".to_string(), "2048".to_string());
        metadata.insert("arch".to_string(), "x86_64".to_string());
        
        let result = chain.evaluate(&metadata).unwrap();
        assert!(result);
        
        // Test failing one filter
        metadata.insert("arch".to_string(), "arm64".to_string());
        let result = chain.evaluate(&metadata).unwrap();
        assert!(!result);
    }

    #[test]
    fn test_invalid_filter() {
        let result = MetadataFilter::parse("size");
        assert!(result.is_err());
    }
}
