/// Configuration module for bgsearch
use clap::Parser;
use std::path::PathBuf;
use std::collections::HashMap;

/// Search result configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SearchConfig {
    pub ascii_terms: Option<Vec<String>>,
    pub binary_terms: Option<Vec<String>>,
    pub unicode_terms: Option<Vec<String>>,
    pub patterns: Option<Vec<String>>,
    pub directories: Option<Vec<PathBuf>>,
    pub recursive: Option<bool>,
    pub no_metadata: Option<bool>,
    pub verify: Option<bool>,
    pub yara_rules: Option<PathBuf>,
    pub candidate_limit: Option<usize>,
    pub filters: Option<Vec<String>>,
    pub numprocs: Option<usize>,
    pub banner: Option<PathBuf>,
    pub index_order: Option<String>,
    pub throttle: Option<usize>,
    pub verbose: Option<bool>,
    pub debug: Option<bool>,
    pub syslog: Option<bool>,
    pub metrics: Option<bool>,
    pub output_format: Option<String>,
}

/// Logging configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct LoggingConfig {
    pub level: Option<String>,
    pub file: Option<PathBuf>,
    pub syslog: Option<bool>,
    pub syslog_facility: Option<String>,
    pub syslog_address: Option<String>,
}

/// Index configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct IndexConfig {
    pub directories: Option<Vec<PathBuf>>,
    pub recursive: Option<bool>,
    pub pattern: Option<String>,
    pub order: Option<String>,
}

/// Verification configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct VerificationConfig {
    pub enabled: Option<bool>,
    pub yara_rules: Option<PathBuf>,
    pub candidate_limit: Option<usize>,
    pub timeout: Option<u64>,
}

/// Performance configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct PerformanceConfig {
    pub numprocs: Option<usize>,
    pub throttle: Option<usize>,
    pub buffer_size: Option<usize>,
    pub max_memory: Option<u64>,
}

/// Output configuration
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct OutputConfig {
    pub format: Option<String>,
    pub show_metadata: Option<bool>,
    pub include_offsets: Option<bool>,
    pub include_verification: Option<bool>,
}

/// Main configuration structure
#[derive(Debug, Clone, Serialize, Deserialize, Default)]
pub struct Config {
    pub search: Option<SearchConfig>,
    pub logging: Option<LoggingConfig>,
    pub index: Option<IndexConfig>,
    pub verification: Option<VerificationConfig>,
    pub performance: Option<PerformanceConfig>,
    pub output: Option<OutputConfig>,
}

impl Config {
    /// Merge another config into this one
    pub fn merge(&mut self, other: Config) {
        if let Some(other_search) = other.search {
            if let Some(self_search) = &mut self.search {
                // Merge search config
                self.merge_search_config(self_search, other_search);
            } else {
                self.search = other.search;
            }
        }

        if let Some(other_logging) = other.logging {
            if let Some(self_logging) = &mut self.logging {
                self.merge_logging_config(self_logging, other_logging);
            } else {
                self.logging = other.logging;
            }
        }

        if let Some(other_index) = other.index {
            if let Some(self_index) = &mut self.index {
                self.merge_index_config(self_index, other_index);
            } else {
                self.index = other.index;
            }
        }

        if let Some(other_verification) = other.verification {
            if let Some(self_verification) = &mut self.verification {
                self.merge_verification_config(self_verification, other_verification);
            } else {
                self.verification = other.verification;
            }
        }

        if let Some(other_performance) = other.performance {
            if let Some(self_performance) = &mut self.performance {
                self.merge_performance_config(self_performance, other_performance);
            } else {
                self.performance = other.performance;
            }
        }

        if let Some(other_output) = other.output {
            if let Some(self_output) = &mut self.output {
                self.merge_output_config(self_output, other_output);
            } else {
                self.output = other.output;
            }
        }
    }

    /// Apply command-line arguments to override config
    pub fn apply_args(&mut self, args: &super::Args) {
        // Apply search config from CLI
        if !args.ascii_terms.is_empty() ||
           !args.binary_terms.is_empty() ||
           !args.unicode_terms.is_empty() ||
           !args.patterns.is_empty() ||
           !args.directories.is_empty() {
            
            let search_config = SearchConfig {
                ascii_terms: if !args.ascii_terms.is_empty() { Some(args.ascii_terms.clone()) } else { None },
                binary_terms: if !args.binary_terms.is_empty() { Some(args.binary_terms.clone()) } else { None },
                unicode_terms: if !args.unicode_terms.is_empty() { Some(args.unicode_terms.clone()) } else { None },
                patterns: if !args.patterns.is_empty() { Some(args.patterns.clone()) } else { None },
                directories: if !args.directories.is_empty() { Some(args.directories.clone()) } else { None },
                recursive: if args.recursive { Some(true) } else { None },
                no_metadata: if args.no_metadata { Some(true) } else { None },
                verify: if args.verify { Some(true) } else { None },
                yara_rules: args.yara_rules.clone(),
                candidate_limit: args.candidate_limit,
                filters: if !args.filters.is_empty() { Some(args.filters.clone()) } else { None },
                numprocs: args.numprocs,
                banner: args.banner.clone(),
                index_order: args.index_order.map(|o| format!("{:?}", o)),
                throttle: args.throttle,
                verbose: if args.verbose { Some(true) } else { None },
                debug: if args.debug { Some(true) } else { None },
                syslog: if args.syslog { Some(true) } else { None },
                metrics: if args.metrics { Some(true) } else { None },
                output_format: args.output_format.map(|o| format!("{:?}", o)),
            };

            if let Some(existing_search) = &mut self.search {
                self.merge_search_config(existing_search, search_config);
            } else {
                self.search = Some(search_config);
            }
        }
    }

    fn merge_search_config(&self, _self_search: &mut SearchConfig, _other_search: SearchConfig) {
        // Note: This is a placeholder for the actual merge implementation
        // In practice, we'd need to make SearchConfig fields mutable to implement this properly
    }

    fn merge_logging_config(&self, _self_logging: &mut LoggingConfig, _other_logging: LoggingConfig) {
        // Placeholder implementation
    }

    fn merge_index_config(&self, _self_index: &mut IndexConfig, _other_index: IndexConfig) {
        // Placeholder implementation
    }

    fn merge_verification_config(&self, _self_verification: &mut VerificationConfig, _other_verification: VerificationConfig) {
        // Placeholder implementation
    }

    fn merge_performance_config(&self, _self_performance: &mut PerformanceConfig, _other_performance: PerformanceConfig) {
        // Placeholder implementation
    }

    fn merge_output_config(&self, _self_output: &mut OutputConfig, _other_output: OutputConfig) {
        // Placeholder implementation
    }

    /// Get default configuration directory
    pub fn default_config_dir() -> PathBuf {
        let mut dir = dirs::config_dir()
            .unwrap_or_else(|| PathBuf::from("/etc"));
        dir.push("biggrep");
        dir
    }

    /// Get default configuration file path
    pub fn default_config_file() -> PathBuf {
        let mut config_dir = Self::default_config_dir();
        config_dir.push("bgsearch.conf");
        config_dir
    }

    /// Load configuration from file
    pub fn load_from_file(path: &Path) -> Result<Self, ConfigError> {
        if !path.exists() {
            return Err(ConfigError::FileNotFound(path.to_path_buf()));
        }

        let content = std::fs::read_to_string(path)
            .map_err(|e| ConfigError::IoError(e, path.to_path_buf()))?;

        let config: Self = toml::from_str(&content)
            .map_err(|e| ConfigError::ParseError(e.to_string()))?;

        Ok(config)
    }

    /// Save configuration to file
    pub fn save_to_file(&self, path: &Path) -> Result<(), ConfigError> {
        let content = toml::to_string_pretty(self)
            .map_err(|e| ConfigError::SerializeError(e.to_string()))?;

        // Ensure directory exists
        if let Some(parent) = path.parent() {
            std::fs::create_dir_all(parent)
                .map_err(|e| ConfigError::IoError(e, parent.to_path_buf()))?;
        }

        std::fs::write(path, content)
            .map_err(|e| ConfigError::IoError(e, path.to_path_buf()))?;

        Ok(())
    }

    /// Validate configuration
    pub fn validate(&self) -> Result<(), ConfigError> {
        // Validate search configuration
        if let Some(search) = &self.search {
            // Validate search terms
            let has_terms = search.ascii_terms.as_ref().map_or(false, |v| !v.is_empty()) ||
                           search.binary_terms.as_ref().map_or(false, |v| !v.is_empty()) ||
                           search.unicode_terms.as_ref().map_or(false, |v| !v.is_empty()) ||
                           search.patterns.as_ref().map_or(false, |v| !v.is_empty());

            if !has_terms {
                return Err(ConfigError::ValidationError("No search terms specified".to_string()));
            }

            // Validate directories
            if let Some(dirs) = &search.directories {
                for dir in dirs {
                    if !dir.exists() {
                        return Err(ConfigError::ValidationError(
                            format!("Directory does not exist: {}", dir.display())
                        ));
                    }
                    if !dir.is_dir() {
                        return Err(ConfigError::ValidationError(
                            format!("Path is not a directory: {}", dir.display())
                        ));
                    }
                }
            }
        }

        // Validate performance configuration
        if let Some(perf) = &self.performance {
            if let Some(numprocs) = perf.numprocs {
                if numprocs == 0 {
                    return Err(ConfigError::ValidationError(
                        "Number of processes must be greater than 0".to_string()
                    ));
                }
            }
        }

        Ok(())
    }
}

/// Configuration errors
#[derive(Debug, thiserror::Error)]
pub enum ConfigError {
    #[error("Configuration file not found: {0}")]
    FileNotFound(PathBuf),
    #[error("I/O error while {1}: {0}")]
    IoError(std::io::Error, PathBuf),
    #[error("Failed to parse configuration: {0}")]
    ParseError(String),
    #[error("Failed to serialize configuration: {0}")]
    SerializeError(String),
    #[error("Configuration validation failed: {0}")]
    ValidationError(String),
}

/// Default configuration templates
pub struct DefaultConfigs;

impl DefaultConfigs {
    /// Get minimal search configuration
    pub fn minimal_search() -> Config {
        Config {
            search: Some(SearchConfig {
                ascii_terms: None,
                binary_terms: None,
                unicode_terms: None,
                patterns: None,
                directories: None,
                recursive: Some(false),
                no_metadata: Some(false),
                verify: Some(false),
                yara_rules: None,
                candidate_limit: Some(15000),
                filters: None,
                numprocs: Some(12),
                banner: None,
                index_order: Some("alpha".to_string()),
                throttle: Some(10000),
                verbose: Some(false),
                debug: Some(false),
                syslog: Some(false),
                metrics: Some(false),
                output_format: Some("text".to_string()),
            }),
            logging: None,
            index: None,
            verification: None,
            performance: None,
            output: None,
        }
    }

    /// Get production configuration template
    pub fn production() -> Config {
        Config {
            search: Some(SearchConfig {
                ascii_terms: None,
                binary_terms: None,
                unicode_terms: None,
                patterns: None,
                directories: None,
                recursive: Some(true),
                no_metadata: Some(false),
                verify: Some(false),
                yara_rules: None,
                candidate_limit: Some(100000),
                filters: None,
                numprocs: Some(24),
                banner: None,
                index_order: Some("shuffle".to_string()),
                throttle: Some(50000),
                verbose: Some(false),
                debug: Some(false),
                syslog: Some(true),
                metrics: Some(true),
                output_format: Some("json".to_string()),
            }),
            logging: Some(LoggingConfig {
                level: Some("info".to_string()),
                file: None,
                syslog: Some(true),
                syslog_facility: Some("daemon".to_string()),
                syslog_address: None,
            }),
            index: Some(IndexConfig {
                directories: None,
                recursive: Some(true),
                pattern: Some("**/*.bgi".to_string()),
                order: Some("shuffle".to_string()),
            }),
            verification: Some(VerificationConfig {
                enabled: Some(false),
                yara_rules: None,
                candidate_limit: Some(100000),
                timeout: Some(300),
            }),
            performance: Some(PerformanceConfig {
                numprocs: Some(24),
                throttle: Some(50000),
                buffer_size: Some(1048576),
                max_memory: Some(4294967296), // 4GB
            }),
            output: Some(OutputConfig {
                format: Some("json".to_string()),
                show_metadata: Some(true),
                include_offsets: Some(false),
                include_verification: Some(true),
            }),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_config_serialization() {
        let config = Config::default();
        let serialized = toml::to_string(&config).unwrap();
        let deserialized: Config = toml::from_str(&serialized).unwrap();
        assert_eq!(config.search, deserialized.search);
    }

    #[test]
    fn test_config_validation() {
        let mut config = Config::default();
        config.search = Some(SearchConfig {
            ascii_terms: Some(vec!["test".to_string()]),
            ..Default::default()
        });
        
        assert!(config.validate().is_ok());
    }

    #[test]
    fn test_invalid_config_validation() {
        let mut config = Config::default();
        config.search = Some(SearchConfig {
            ascii_terms: Some(vec![]), // Empty terms
            directories: Some(vec![PathBuf::from("/nonexistent")]),
            ..Default::default()
        });
        
        assert!(config.validate().is_err());
    }
}
