#!/bin/bash
# Test script for rs-bgindex
# This script demonstrates the usage of rs-bgindex with various scenarios

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}======================================${NC}"
echo -e "${BLUE}rs-bgindex Test Suite${NC}"
echo -e "${BLUE}======================================${NC}\n"

# Check if Rust is installed
if ! command -v cargo &> /dev/null; then
    echo -e "${RED}Error: cargo not found. Please install Rust to run tests.${NC}"
    echo "Visit: https://rustup.rs/"
    exit 1
fi

# Build the project
echo -e "${YELLOW}Building rs-bgindex...${NC}"
cd "$(dirname "$0")/../.."
cargo build --package rs-bgindex --release

if [ $? -ne 0 ]; then
    echo -e "${RED}Build failed!${NC}"
    exit 1
fi

echo -e "${GREEN}Build successful!${NC}\n"

# Test 1: Basic 3-gram indexing
echo -e "${YELLOW}Test 1: Basic 3-gram indexing${NC}"
echo "Creating test files..."
mkdir -p test_data
echo "This is a test file for the BigGrep indexer. It contains various patterns." > test_data/file1.txt
echo "Another test file with different content but similar patterns." > test_data/file2.txt
echo "Third file with completely different content to test indexing." > test_data/file3.txt

echo "Creating file list..."
ls -1 test_data/ | sed 's/^/0:test_data\//' > test_file_list.txt

echo -e "${BLUE}Running basic 3-gram indexing...${NC}"
./target/release/rs-bgindex -p test_basic_3gram -v < test_file_list.txt

if [ $? -eq 0 ]; then
    echo -e "${GREEN}Test 1 PASSED${NC}"
    ls -lh test_basic_3gram.bgi 2>/dev/null || echo "Index file not created"
else
    echo -e "${RED}Test 1 FAILED${NC}"
    exit 1
fi

echo ""

# Test 2: 4-gram indexing
echo -e "${YELLOW}Test 2: 4-gram indexing${NC}"
echo -e "${BLUE}Running 4-gram indexing...${NC}"
./target/release/rs-bgindex -n 4 -p test_4gram -v < test_file_list.txt

if [ $? -eq 0 ]; then
    echo -e "${GREEN}Test 2 PASSED${NC}"
    ls -lh test_4gram.bgi 2>/dev/null || echo "Index file not created"
else
    echo -e "${RED}Test 2 FAILED${NC}"
    exit 1
fi

echo ""

# Test 3: High-performance mode with multiple threads
echo -e "${YELLOW}Test 3: High-performance mode${NC}"
echo -e "${BLUE}Running with 8 shingling and 8 compression threads...${NC}"
./target/release/rs-bgindex \
    -S 8 \
    -C 8 \
    -L \
    -p test_high_perf \
    -v < test_file_list.txt

if [ $? -eq 0 ]; then
    echo -e "${GREEN}Test 3 PASSED${NC}"
    ls -lh test_high_perf.bgi 2>/dev/null || echo "Index file not created"
else
    echo -e "${RED}Test 3 FAILED${NC}"
    exit 1
fi

echo ""

# Test 4: Overflow handling
echo -e "${YELLOW}Test 4: Overflow handling${NC}"
echo "Creating a file with many unique N-grams..."
dd if=/dev/urandom of=test_data/large_random.bin bs=1M count=10 2>/dev/null

echo "0:test_data/large_random.bin" >> test_file_list.txt

echo -e "${BLUE}Running with max unique N-gram limit...${NC}"
./target/release/rs-bgindex \
    -M 100000 \
    -O overflow_list.txt \
    -p test_overflow \
    -v < test_file_list.txt 2>&1 | tee overflow.log || true

if [ -f overflow_list.txt ] && [ -s overflow_list.txt ]; then
    echo -e "${GREEN}Overflow handling test - overflow file created${NC}"
    echo -e "${BLUE}Overflow contents:${NC}"
    cat overflow_list.txt
else
    echo -e "${YELLOW}No overflow detected (file may not exceed limit)${NC}"
fi

echo ""

# Test 5: Different compression settings
echo -e "${YELLOW}Test 5: Custom compression settings${NC}"
echo -e "${BLUE}Running with custom PFOR blocksize and exceptions...${NC}"
./target/release/rs-bgindex \
    -b 64 \
    -e 4 \
    -m 8 \
    -p test_custom_compression \
    -v < test_file_list.txt

if [ $? -eq 0 ]; then
    echo -e "${GREEN}Test 5 PASSED${NC}"
    ls -lh test_custom_compression.bgi 2>/dev/null || echo "Index file not created"
else
    echo -e "${RED}Test 5 FAILED${NC}"
    exit 1
fi

echo ""

# Test 6: Logging
echo -e "${YELLOW}Test 6: Logging functionality${NC}"
echo -e "${BLUE}Running with logging to file...${NC}"
./target/release/rs-bgindex \
    -p test_logging \
    -l test_build.log \
    -d \
    -v < test_file_list.txt

if [ $? -eq 0 ] && [ -f test_build.log ]; then
    echo -e "${GREEN}Test 6 PASSED${NC}"
    echo -e "${BLUE}Log file created with $(wc -l < test_build.log) lines${NC}"
else
    echo -e "${RED}Test 6 FAILED${NC}"
    exit 1
fi

echo ""

# Test 7: Help and version
echo -e "${YELLOW}Test 7: Help and version information${NC}"
echo -e "${BLUE}Testing help flag...${NC}"
./target/release/rs-bgindex --help > /dev/null 2>&1
if [ $? -eq 0 ]; then
    echo -e "${GREEN}Help flag works${NC}"
else
    echo -e "${RED}Help flag failed${NC}"
fi

echo -e "${BLUE}Testing version flag...${NC}"
./target/release/rs-bgindex --version > /dev/null 2>&1
if [ $? -eq 0 ]; then
    echo -e "${GREEN}Version flag works${NC}"
else
    echo -e "${RED}Version flag failed${NC}"
fi

echo ""

# Cleanup
echo -e "${YELLOW}Cleaning up test files...${NC}"
rm -rf test_data test_file_list.txt overflow_list.txt overflow.log

# Summary
echo -e "${BLUE}======================================${NC}"
echo -e "${GREEN}All tests completed successfully!${NC}"
echo -e "${BLUE}======================================${NC}"
echo ""
echo "The following test indexes were created (if tests passed):"
ls -lh *.bgi 2>/dev/null || echo "No index files found in current directory"
