//! Utility functions for BigGrep

use crate::{BigGrepResult};
use std::path::{Path, PathBuf};
use std::collections::HashMap;

/// Configuration for BigGrep operations
#[derive(Debug, Clone)]
pub struct Config {
    pub max_file_size: u64,
    pub num_threads: usize,
    pub buffer_size: usize,
    pub use_memory_mapping: bool,
    pub log_level: String,
}

impl Default for Config {
    fn default() -> Self {
        Self {
            max_file_size: 100 * 1024 * 1024, // 100MB
            num_threads: rayon::current_num_threads(),
            buffer_size: 8 * 1024, // 8KB
            use_memory_mapping: true,
            log_level: "info".to_string(),
        }
    }
}

/// File utility functions
pub struct FileUtils;

impl FileUtils {
    /// Check if a file should be processed based on size and type
    pub fn should_process_file(path: &Path, config: &Config) -> bool {
        if let Ok(metadata) = std::fs::metadata(path) {
            // Skip files larger than max_file_size
            if metadata.len() > config.max_file_size {
                return false;
            }
            
            // Skip files that are directories
            if metadata.is_dir() {
                return false;
            }
            
            return true;
        }
        false
    }
    
    /// Get file extension
    pub fn get_extension(path: &Path) -> Option<String> {
        path.extension()
            .and_then(|ext| ext.to_str())
            .map(|s| s.to_lowercase())
    }
    
    /// Check if file is a text file based on extension
    pub fn is_text_file(path: &Path) -> bool {
        match Self::get_extension(path) {
            Some(ext) => {
                matches!(ext.as_str(), 
                    "txt" | "md" | "rs" | "py" | "js" | "ts" | "java" | "cpp" | "c" | "h" | 
                    "go" | "rs" | "php" | "rb" | "sh" | "bat" | "yml" | "yaml" | "json" | "xml" | 
                    "toml" | "ini" | "cfg" | "conf" | "log"
                )
            }
            None => true, // Default to text if no extension
        }
    }
    
    /// Read file with memory mapping if enabled
    pub fn read_file(path: &Path, use_mmap: bool, buffer_size: usize) -> BigGrepResult<Vec<u8>> {
        if use_mmap {
            Self::read_file_mmap(path)
        } else {
            Self::read_file_buffered(path, buffer_size)
        }
    }
    
    /// Read file using memory mapping
    fn read_file_mmap(path: &Path) -> BigGrepResult<Vec<u8>> {
        use memmap2::Mmap;
        
        let file = std::fs::File::open(path)?;
        let mmap = unsafe { Mmap::map(&file)? };
        Ok(mmap.to_vec())
    }
    
    /// Read file using buffered I/O
    fn read_file_buffered(path: &Path, buffer_size: usize) -> BigGrepResult<Vec<u8>> {
        use std::io::{BufRead, BufReader, Read};
        
        let file = std::fs::File::open(path)?;
        let reader = BufReader::with_capacity(buffer_size, file);
        
        let mut content = Vec::new();
        for line in reader.lines() {
            let line = line?;
            content.extend_from_slice(line.as_bytes());
            content.push(b'\n');
        }
        
        Ok(content)
    }
    
    /// Walk directory and collect all files
    pub fn collect_files(dir: &Path) -> BigGrepResult<Vec<PathBuf>> {
        let mut files = Vec::new();
        
        if dir.is_dir() {
            for entry in std::fs::read_dir(dir)? {
                let entry = entry?;
                let path = entry.path();
                
                if path.is_dir() {
                    files.extend(Self::collect_files(&path)?);
                } else if path.is_file() {
                    files.push(path);
                }
            }
        } else if dir.is_file() {
            files.push(dir.to_path_buf());
        }
        
        Ok(files)
    }
    
    /// Calculate file hash
    pub fn calculate_file_hash(path: &Path, algorithm: &str) -> BigGrepResult<String> {
        use std::io::{BufRead, BufReader};
        
        let file = std::fs::File::open(path)?;
        let reader = BufReader::new(file);
        
        match algorithm {
            "simple" => {
                let mut hasher = std::collections::hash_map::DefaultHasher::new();
                
                for line in reader.lines() {
                    let line = line?;
                    hasher.write(line.as_bytes());
                    hasher.write(&[b'\n']);
                }
                
                Ok(format!("{:x}", hasher.finish()))
            }
            _ => {
                Err(anyhow::anyhow!("Unsupported hash algorithm: {}", algorithm))
            }
        }
    }
}

/// Parallel processing utilities
pub struct ParallelUtils;

impl ParallelUtils {
    /// Process files in parallel
    pub fn process_files_parallel<F, R>(
        files: &[PathBuf], 
        num_threads: usize,
        processor: F
    ) -> BigGrepResult<Vec<R>>
    where
        F: Fn(&PathBuf) -> BigGrepResult<R> + Send + Sync,
        R: Send,
    {
        use rayon::prelude::*;
        
        let results: Vec<R> = files
            .par_iter()
            .with_max_len(1)
            .map(|file| processor(file))
            .collect::<Result<Vec<R>, _>>()?;
        
        Ok(results)
    }
    
    /// Process chunks of data in parallel
    pub fn process_chunks_parallel<F, R, T>(
        data: &[T], 
        chunk_size: usize,
        processor: F
    ) -> BigGrepResult<Vec<R>>
    where
        F: Fn(&[T]) -> BigGrepResult<R> + Send + Sync,
        R: Send,
        T: Send + Sync,
    {
        use rayon::prelude::*;
        
        let chunks: Vec<&[T]> = data.chunks(chunk_size).collect();
        let results: Vec<R> = chunks
            .par_iter()
            .map(|chunk| processor(chunk))
            .collect::<Result<Vec<R>, _>>()?;
        
        Ok(results)
    }
}

/// Time utilities
pub struct TimeUtils;

impl TimeUtils {
    /// Format duration in human readable format
    pub fn format_duration(duration: std::time::Duration) -> String {
        let secs = duration.as_secs();
        
        if secs < 60 {
            format!("{:.1}s", secs as f64)
        } else if secs < 3600 {
            let minutes = secs / 60;
            let remaining_secs = secs % 60;
            format!("{}m{}s", minutes, if remaining_secs > 0 { format!("{}", remaining_secs) } else { "".to_string() })
        } else {
            let hours = secs / 3600;
            let remaining = secs % 3600;
            let minutes = remaining / 60;
            format!("{}h{}m", hours, minutes)
        }
    }
    
    /// Get current timestamp
    pub fn now() -> u64 {
        std::time::SystemTime::now()
            .duration_since(std::time::UNIX_EPOCH)
            .unwrap()
            .as_secs()
    }
}