//! Search module for BigGrep
//! 
//! Contains search algorithms and utilities for fast pattern matching.

pub mod boyer_moore;

// Re-export Boyer-Moore-Horspool types
pub use boyer_moore::{
    BoyerMooreHorspool, 
    SearchResult, 
    VerificationResult,
    utils
};

/// Search engine configuration
#[derive(Debug, Clone)]
pub struct SearchOptions {
    pub case_sensitive: bool,
    pub whole_word: bool,
    pub show_line_numbers: bool,
    pub show_context: bool,
    pub context_lines: usize,
    pub max_results: Option<usize>,
    pub binary_files: bool,
}

impl Default for SearchOptions {
    fn default() -> Self {
        Self {
            case_sensitive: false,
            whole_word: false,
            show_line_numbers: true,
            show_context: false,
            context_lines: 3,
            max_results: None,
            binary_files: false,
        }
    }
}

/// Search result with detailed information
#[derive(Debug, Clone)]
pub struct SearchResult {
    pub matches: Vec<SearchMatch>,
    pub total_matches: usize,
    pub files_searched: usize,
    pub search_duration_ms: u64,
}

/// Individual search match
#[derive(Debug, Clone)]
pub struct SearchMatch {
    pub file_path: std::path::PathBuf,
    pub line_number: Option<usize>,
    pub column: Option<usize>,
    pub match_text: String,
    pub context_before: Vec<String>,
    pub context_after: Vec<String>,
}

/// Search candidate file
#[derive(Debug, Clone)]
pub struct Candidate {
    pub file_path: std::path::PathBuf,
    pub score: f64,
    pub metadata: Option<String>,
}

/// Search engine for pattern matching
#[derive(Debug)]
pub struct SearchEngine {
    patterns: Vec<String>,
    options: SearchOptions,
}

impl SearchEngine {
    /// Create new search engine
    pub fn new(patterns: Vec<String>, options: SearchOptions) -> Self {
        Self { patterns, options }
    }
    
    /// Search in memory buffer
    pub fn search_memory(&self, data: &[u8], file_path: &std::path::Path) -> Result<Vec<SearchMatch>, String> {
        // Placeholder implementation
        // This would use the Boyer-Moore-Horspool implementation
        Ok(Vec::new())
    }
    
    /// Search in file
    pub fn search_file(&self, file_path: &std::path::Path, content: &str) -> Result<Vec<SearchMatch>, String> {
        // Placeholder implementation
        // This would use the Boyer-Moore-Horspool implementation
        Ok(Vec::new())
    }
}
