//! Metadata module for file-level metadata handling
//! 
//! Provides metadata storage and retrieval for files including paths, sizes,
//! checksums, and optional byte-range mappings for extraction.

use crate::error::{BigGrepError, BigGrepResult};
use std::collections::{HashMap, BTreeMap};
use std::fs;
use std::io::{Read, Write, Seek, SeekFrom};
use std::path::{Path, PathBuf};
use std::time::{SystemTime, UNIX_EPOCH};
use serde::{Deserialize, Serialize};
use byteorder::{ReadBytesExt, WriteBytesExt, BigEndian};

/// File-level metadata record
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct FileMetadata {
    pub file_id: u64,
    pub path: PathBuf,
    pub size_bytes: u64,
    pub checksum: Option<String>,
    pub created_at: u64,
    pub modified_at: u64,
    pub byte_range_map: Option<Vec<ByteRange>>,
    pub mime_type: Option<String>,
    pub encoding: Option<String>,
}

impl FileMetadata {
    pub fn new(file_path: PathBuf) -> BigGrepResult<Self> {
        let metadata = fs::metadata(&file_path)
            .map_err(|e| BigGrepError::Io(e))?;
        
        let created_at = metadata.created()
            .unwrap_or(SystemTime::UNIX_EPOCH)
            .duration_since(UNIX_EPOCH)
            .unwrap()
            .as_secs();
            
        let modified_at = metadata.modified()
            .unwrap_or(SystemTime::UNIX_EPOCH)
            .duration_since(UNIX_EPOCH)
            .unwrap()
            .as_secs();
        
        Ok(Self {
            file_id: 0,
            path: file_path,
            size_bytes: metadata.len(),
            checksum: None,
            created_at,
            modified_at,
            byte_range_map: None,
            mime_type: None,
            encoding: None,
        })
    }
    
    pub fn with_checksum(mut self, checksum: String) -> Self {
        self.checksum = Some(checksum);
        self
    }
    
    pub fn with_byte_ranges(mut self, ranges: Vec<ByteRange>) -> Self {
        self.byte_range_map = Some(ranges);
        self
    }
    
    pub fn with_mime_type(mut self, mime_type: String) -> Self {
        self.mime_type = Some(mime_type);
        self
    }
}

/// Byte range mapping for file extraction
#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct ByteRange {
    pub start: u64,
    pub end: u64,
    pub token_range: Option<(usize, usize)>,
}

impl ByteRange {
    pub fn new(start: u64, end: u64) -> Self {
        Self {
            start,
            end,
            token_range: None,
        }
    }
    
    pub fn with_token_range(mut self, start_token: usize, end_token: usize) -> Self {
        self.token_range = Some((start_token, end_token));
        self
    }
    
    pub fn length(&self) -> u64 {
        self.end - self.start
    }
    
    pub fn contains_byte(&self, byte: u64) -> bool {
        byte >= self.start && byte < self.end
    }
}

/// Metadata store for managing file metadata
#[derive(Debug)]
pub struct MetadataStore {
    files: HashMap<u64, FileMetadata>,
    path_index: HashMap<PathBuf, u64>,
    next_id: u64,
}

impl MetadataStore {
    pub fn new() -> Self {
        Self {
            files: HashMap::new(),
            path_index: HashMap::new(),
            next_id: 1,
        }
    }
    
    pub fn add_file(&mut self, file_path: PathBuf) -> BigGrepResult<u64> {
        let metadata = FileMetadata::new(file_path)?;
        let file_id = self.next_id;
        self.next_id += 1;
        
        let mut metadata = metadata;
        metadata.file_id = file_id;
        
        self.files.insert(file_id, metadata.clone());
        self.path_index.insert(metadata.path.clone(), file_id);
        
        Ok(file_id)
    }
    
    pub fn get_by_id(&self, file_id: u64) -> Option<&FileMetadata> {
        self.files.get(&file_id)
    }
    
    pub fn get_by_path(&self, path: &Path) -> Option<&FileMetadata> {
        self.path_index.get(path).and_then(|id| self.files.get(id))
    }
    
    pub fn files(&self) -> impl Iterator<Item = &FileMetadata> + '_ {
        self.files.values()
    }
    
    pub fn stats(&self) -> StoreStats {
        StoreStats {
            file_count: self.files.len(),
            total_bytes: self.files.values().map(|m| m.size_bytes).sum(),
            files_with_checksums: self.files.values().filter(|m| m.checksum.is_some()).count(),
            files_with_byte_ranges: self.files.values().filter(|m| m.byte_range_map.is_some()).count(),
        }
    }
}

/// Metadata store statistics
#[derive(Debug, Clone)]
pub struct StoreStats {
    pub file_count: usize,
    pub total_bytes: u64,
    pub files_with_checksums: usize,
    pub files_with_byte_ranges: usize,
}

impl StoreStats {
    pub fn avg_file_size(&self) -> f64 {
        if self.file_count == 0 {
            0.0
        } else {
            self.total_bytes as f64 / self.file_count as f64
        }
    }
}

/// Metadata builder for creating metadata with various options
#[derive(Debug)]
pub struct MetadataBuilder {
    include_checksums: bool,
    include_byte_ranges: bool,
    include_mime_type: bool,
}

impl MetadataBuilder {
    pub fn new() -> Self {
        Self {
            include_checksums: false,
            include_byte_ranges: false,
            include_mime_type: false,
        }
    }
    
    pub fn with_checksums(mut self) -> Self {
        self.include_checksums = true;
        self
    }
    
    pub fn with_byte_ranges(mut self) -> Self {
        self.include_byte_ranges = true;
        self
    }
    
    pub fn with_mime_type(mut self) -> Self {
        self.include_mime_type = true;
        self
    }
    
    pub fn build_for_file(&self, file_path: &Path) -> BigGrepResult<FileMetadata> {
        let mut metadata = FileMetadata::new(file_path.to_path_buf())?;
        
        if self.include_checksums {
            let checksum = self.calculate_checksum(file_path)?;
            metadata = metadata.with_checksum(checksum);
        }
        
        if self.include_mime_type {
            let mime_type = self.detect_mime_type(file_path);
            metadata = metadata.with_mime_type(mime_type);
        }
        
        if self.include_byte_ranges {
            let ranges = vec![ByteRange::new(0, metadata.size_bytes)];
            metadata = metadata.with_byte_ranges(ranges);
        }
        
        Ok(metadata)
    }
    
    fn calculate_checksum(&self, file_path: &Path) -> BigGrepResult<String> {
        use std::collections::hash_map::DefaultHasher;
        use std::hash::{Hash, Hasher};
        
        let file = fs::File::open(file_path)
            .map_err(|e| BigGrepError::Io(e))?;
        
        let mut hasher = DefaultHasher::new();
        let metadata = fs::metadata(file_path)
            .map_err(|e| BigGrepError::Io(e))?;
        let size = metadata.len();
        
        if size > 0 {
            let to_read = std::cmp::min(1024, size);
            let mut buffer = vec![0u8; to_read as usize];
            let mut reader = std::io::BufReader::new(&file);
            reader.read_exact(&mut buffer)?;
            hasher.write(&buffer);
        }
        
        Ok(format!("{:016x}", hasher.finish()))
    }
    
    fn detect_mime_type(&self, file_path: &Path) -> String {
        let ext = file_path.extension()
            .and_then(|e| e.to_str())
            .unwrap_or("")
            .to_lowercase();
        
        match ext.as_str() {
            "txt" | "md" | "rst" => "text/plain",
            "html" | "htm" => "text/html",
            "json" => "application/json",
            _ => "application/octet-stream",
        }.to_string()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use tempfile::NamedTempFile;
    
    #[test]
    fn test_metadata_store() {
        let mut store = MetadataStore::new();
        let temp_file = NamedTempFile::new().unwrap();
        let path = temp_file.path().to_path_buf();
        
        let file_id = store.add_file(path.clone()).unwrap();
        assert_eq!(file_id, 1);
        
        let metadata = store.get_by_id(file_id).unwrap();
        assert_eq!(metadata.file_id, 1);
        assert_eq!(metadata.path, path);
    }
    
    #[test]
    fn test_metadata_builder() {
        let builder = MetadataBuilder::new()
            .with_checksums()
            .with_mime_type();
        
        let temp_file = NamedTempFile::new().unwrap();
        let path = temp_file.path().to_path_buf();
        
        let metadata = builder.build_for_file(&path).unwrap();
        assert!(metadata.checksum.is_some());
        assert!(metadata.mime_type.is_some());
    }
}
