//! File I/O utilities for BigGrep operations
//! 
//! Provides memory-mapped file access, file processing utilities,
//! and streaming I/O operations for large file handling.

use crate::error::{BigGrepError, BigGrepResult};
use memmap2::{Mmap, MmapOptions};
use std::fs::File;
use std::io::{Read, Write, BufReader, BufWriter};
use std::path::{Path, PathBuf};

/// Memory-mapped file wrapper
#[derive(Debug)]
pub struct MemoryMappedFile {
    mmap: Mmap,
    path: PathBuf,
    size: u64,
}

impl MemoryMappedFile {
    pub fn open(path: &Path) -> BigGrepResult<Self> {
        let file = File::open(path).map_err(|e| BigGrepError::Io(e))?;
        let size = file.metadata().map_err(|e| BigGrepError::Io(e))?.len();
        let mmap = unsafe { MmapOptions::new().map(&file).map_err(|e| BigGrepError::MemoryMap(e.to_string()))? };
        Ok(Self { mmap, path: path.to_path_buf(), size })
    }
    
    pub fn as_bytes(&self) -> &[u8] { &self.mmap }
    pub fn size(&self) -> u64 { self.size }
    pub fn path(&self) -> &Path { &self.path }
}

/// File processor for batch operations
#[derive(Debug)]
pub struct FileProcessor {
    chunk_size: usize,
    use_memory_mapping: bool,
}

impl FileProcessor {
    pub fn new() -> Self {
        Self {
            chunk_size: 1024 * 1024,
            use_memory_mapping: true,
        }
    }
    
    pub fn with_chunk_size(mut self, size: usize) -> Self {
        self.chunk_size = size.max(1024);
        self
    }
    
    pub fn with_memory_mapping(mut self, enabled: bool) -> Self {
        self.use_memory_mapping = enabled;
        self
    }
    
    pub fn process_file<F, R>(&self, file_path: &Path, processor: F) -> BigGrepResult<R>
    where
        F: Fn(&[u8]) -> BigGrepResult<R>,
        R: Send,
    {
        if self.use_memory_mapping {
            let mmap = MemoryMappedFile::open(file_path)?;
            processor(mmap.as_bytes())
        } else {
            let file = File::open(file_path).map_err(|e| BigGrepError::Io(e))?;
            let mut reader = BufReader::new(file);
            let mut content = String::new();
            reader.read_to_string(&mut content).map_err(|e| BigGrepError::Io(e))?;
            processor(content.as_bytes())
        }
    }
    
    pub fn write_file(&self, file_path: &Path, data: &[u8]) -> BigGrepResult<()> {
        let file = File::create(file_path).map_err(|e| BigGrepError::Io(e))?;
        let mut writer = BufWriter::new(file);
        writer.write_all(data).map_err(|e| BigGrepError::Io(e))?;
        Ok(())
    }
}

impl Default for FileProcessor {
    fn default() -> Self {
        Self::new()
    }
}

/// File utilities
pub struct FileUtils;

impl FileUtils {
    pub fn is_readable(path: &Path) -> bool {
        fs::metadata(path).map(|m| m.permissions().readonly() != Ok(true)).unwrap_or(false)
    }
    
    pub fn file_size(path: &Path) -> BigGrepResult<u64> {
        fs::metadata(path).map_err(|e| BigGrepError::Io(e)).map(|m| m.len())
    }
    
    pub fn is_text_file(path: &Path) -> BigGrepResult<bool> {
        let file = File::open(path).map_err(|e| BigGrepError::Io(e))?;
        let mut reader = BufReader::new(file);
        let mut buffer = vec![0u8; 1024];
        let bytes_read = reader.read(&mut buffer).map_err(|e| BigGrepError::Io(e))?;
        if bytes_read == 0 { return Ok(true); }
        let sample = &buffer[..bytes_read];
        let null_count = sample.iter().filter(|&&b| b == 0).count();
        let high_bit_count = sample.iter().filter(|&&b| b >= 0x80).count();
        let threshold = bytes_read as f64 * 0.3;
        Ok(null_count as f64 <= threshold && high_bit_count as f64 <= threshold)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use tempfile::NamedTempFile;
    
    #[test]
    fn test_memory_mapped_file() {
        let temp_file = NamedTempFile::new().unwrap();
        let path = temp_file.path().to_path_buf();
        let mmap_file = MemoryMappedFile::open(&path).unwrap();
        assert_eq!(mmap_file.size(), 0);
    }
    
    #[test]
    fn test_file_processor() {
        let processor = FileProcessor::new();
        let temp_file = NamedTempFile::new().unwrap();
        let path = temp_file.path().to_path_buf();
        processor.write_file(&path, b"test content").unwrap();
        let result = processor.process_file(&path, |data| Ok(String::from_utf8_lossy(data).to_string())).unwrap();
        assert_eq!(result, "test content");
    }
    
    #[test]
    fn test_file_utils() {
        let temp_file = NamedTempFile::new().unwrap();
        let path = temp_file.path().to_path_buf();
        assert!(FileUtils::is_readable(&path));
        let size = FileUtils::file_size(&path).unwrap();
        assert_eq!(size, 0);
    }
}
