//! Error handling for BigGrep

use std::fmt;
use std::io;
use std::path::PathBuf;

/// BigGrep specific error type
#[derive(Debug)]
pub enum BigGrepError {
    IoError(io::Error),
    RegexError(regex::Error),
    SerializationError(serde_json::Error),
    Index(String),
    Deserialization(String),
    Serialization(String),
    Io(io::Error),
    SearchError(String),
    ParseError(String),
    VerifyError(String),
    ConfigError(String),
    FileTooLarge,
    InvalidIndex,
    NoResultsFound,
}

pub type BigGrepResult<T> = std::result::Result<T, BigGrepError>;

impl fmt::Display for BigGrepError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            BigGrepError::IoError(e) => write!(f, "I/O error: {}", e),
            BigGrepError::RegexError(e) => write!(f, "Regex error: {}", e),
            BigGrepError::SerializationError(e) => write!(f, "Serialization error: {}", e),
            BigGrepError::Index(msg) => write!(f, "Index error: {}", msg),
            BigGrepError::Deserialization(msg) => write!(f, "Deserialization error: {}", msg),
            BigGrepError::Serialization(msg) => write!(f, "Serialization error: {}", msg),
            BigGrepError::Io(e) => write!(f, "I/O error: {}", e),
            BigGrepError::SearchError(msg) => write!(f, "Search error: {}", msg),
            BigGrepError::ParseError(msg) => write!(f, "Parse error: {}", msg),
            BigGrepError::VerifyError(msg) => write!(f, "Verification error: {}", msg),
            BigGrepError::ConfigError(msg) => write!(f, "Configuration error: {}", msg),
            BigGrepError::FileTooLarge => write!(f, "File too large to process"),
            BigGrepError::InvalidIndex => write!(f, "Invalid or corrupted index"),
            BigGrepError::NoResultsFound => write!(f, "No results found"),
        }
    }
}

impl std::error::Error for BigGrepError {}

impl From<io::Error> for BigGrepError {
    fn from(error: io::Error) -> Self {
        BigGrepError::IoError(error)
    }
}

impl From<regex::Error> for BigGrepError {
    fn from(error: regex::Error) -> Self {
        BigGrepError::RegexError(error)
    }
}

impl From<serde_json::Error> for BigGrepError {
    fn from(error: serde_json::Error) -> Self {
        BigGrepError::SerializationError(error)
    }
}

impl From<std::io::Error> for BigGrepError {
    fn from(error: std::io::Error) -> Self {
        BigGrepError::Io(error)
    }
}